'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
function noop() {}

/**
 * When click and hold on a button - the speed of auto changin the value.
 */
var SPEED = 200;

/**
 * When click and hold on a button - the delay before auto changin the value.
 */
var DELAY = 600;

exports["default"] = {
  getDefaultProps: function getDefaultProps() {
    return {
      max: Infinity,
      min: -Infinity,
      step: 1,
      style: {},
      defaultValue: '',
      onChange: noop,
      onKeyDown: noop,
      onFocus: noop,
      onBlur: noop
    };
  },
  getInitialState: function getInitialState() {
    var value = void 0;
    var props = this.props;
    if ('value' in props) {
      value = props.value;
    } else {
      value = props.defaultValue;
    }
    value = this.toNumber(value);
    return {
      inputValue: this.toPrecisionAsStep(value),
      value: value,
      focused: props.autoFocus
    };
  },
  componentWillReceiveProps: function componentWillReceiveProps(nextProps) {
    if ('value' in nextProps) {
      this.setState({
        inputValue: nextProps.value,
        value: nextProps.value
      });
    }
  },
  componentWillUnmount: function componentWillUnmount() {
    this.stop();
  },
  onChange: function onChange(e) {
    var input = this.getValueFromEvent(e).trim();
    this.setState({ inputValue: input });
    this.props.onChange(this.toNumberWhenUserInput(input)); // valid number or invalid string
  },
  onFocus: function onFocus() {
    var _props;

    this.setState({
      focused: true
    });
    (_props = this.props).onFocus.apply(_props, arguments);
  },
  onBlur: function onBlur(e) {
    var _this = this;

    for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      args[_key - 1] = arguments[_key];
    }

    this.setState({
      focused: false
    });
    var value = this.getCurrentValidValue(this.state.inputValue);
    this.setValue(value, function () {
      var _props2;

      (_props2 = _this.props).onBlur.apply(_props2, [e].concat(args));
    });
  },
  getCurrentValidValue: function getCurrentValidValue(value) {
    var val = value;
    var props = this.props;
    if (val === '') {
      val = '';
    } else if (!this.isNotCompleteNumber(val)) {
      val = Number(val);
      if (val < props.min) {
        val = props.min;
      }
      if (val > props.max) {
        val = props.max;
      }
    } else {
      val = this.state.value;
    }
    return this.toNumber(val);
  },
  setValue: function setValue(v, callback) {
    // trigger onChange
    var newValue = this.isNotCompleteNumber(parseFloat(v, 10)) ? undefined : parseFloat(v, 10);
    var changed = newValue !== this.state.value;
    if (!('value' in this.props)) {
      this.setState({
        value: v,
        inputValue: this.toPrecisionAsStep(v)
      }, callback);
    } else {
      // always set input value same as value
      this.setState({
        inputValue: this.toPrecisionAsStep(this.state.value)
      }, callback);
    }
    if (changed) {
      this.props.onChange(newValue);
    }
  },
  getPrecision: function getPrecision(value) {
    var valueString = value.toString();
    if (valueString.indexOf('e-') >= 0) {
      return parseInt(valueString.slice(valueString.indexOf('e-') + 1), 10);
    }
    var precision = 0;
    if (valueString.indexOf('.') >= 0) {
      precision = valueString.length - valueString.indexOf('.') - 1;
    }
    return precision;
  },


  // step={1.0} value={1.51}
  // press +
  // then value should be 2.51, rather than 2.5
  // https://github.com/react-component/input-number/issues/39
  getMaxPrecision: function getMaxPrecision(currentValue) {
    var step = this.props.step;

    var stepPrecision = this.getPrecision(step);
    if (!currentValue) {
      return stepPrecision;
    }
    var currentValuePrecision = this.getPrecision(currentValue);
    return currentValuePrecision > stepPrecision ? currentValuePrecision : stepPrecision;
  },
  getPrecisionFactor: function getPrecisionFactor(currentValue) {
    var precision = this.getMaxPrecision(currentValue);
    return Math.pow(10, precision);
  },
  toPrecisionAsStep: function toPrecisionAsStep(num) {
    if (this.isNotCompleteNumber(num) || num === '') {
      return num;
    }
    var precision = Math.abs(this.getMaxPrecision(num));
    if (precision) {
      return Number(num).toFixed(precision);
    }
    return num.toString();
  },


  // '1.' '1x' 'xx' '' => are not complete numbers
  isNotCompleteNumber: function isNotCompleteNumber(num) {
    return isNaN(num) || num === '' || num.toString().indexOf('.') === num.toString().length - 1;
  },
  toNumber: function toNumber(num) {
    if (this.isNotCompleteNumber(num)) {
      return num;
    }
    return Number(num);
  },


  // '1.0' '1.00'  => may be a inputing number
  toNumberWhenUserInput: function toNumberWhenUserInput(num) {
    if (/\.0+$/.test(num) && this.state.focused) {
      return num;
    }
    return this.toNumber(num);
  },
  upStep: function upStep(val) {
    var _props3 = this.props,
        step = _props3.step,
        min = _props3.min;

    var precisionFactor = this.getPrecisionFactor(val);
    var precision = Math.abs(this.getMaxPrecision(val));
    var result = void 0;
    if (typeof val === 'number') {
      result = ((precisionFactor * val + precisionFactor * step) / precisionFactor).toFixed(precision);
    } else {
      result = min === -Infinity ? step : min;
    }
    return this.toNumber(result);
  },
  downStep: function downStep(val) {
    var _props4 = this.props,
        step = _props4.step,
        min = _props4.min;

    var precisionFactor = this.getPrecisionFactor(val);
    var precision = Math.abs(this.getMaxPrecision(val));
    var result = void 0;
    if (typeof val === 'number') {
      result = ((precisionFactor * val - precisionFactor * step) / precisionFactor).toFixed(precision);
    } else {
      result = min === -Infinity ? -step : min;
    }
    return this.toNumber(result);
  },
  step: function step(type, e) {
    if (e) {
      e.preventDefault();
    }
    var props = this.props;
    if (props.disabled) {
      return;
    }
    var value = this.getCurrentValidValue(this.state.inputValue);
    if (this.isNotCompleteNumber(value)) {
      return;
    }
    var val = this[type + 'Step'](value);
    if (val > props.max || val < props.min) {
      return;
    }
    this.setValue(val);
    this.setState({
      focused: true
    });
  },
  stop: function stop() {
    if (this.autoStepTimer) {
      clearTimeout(this.autoStepTimer);
    }
  },
  down: function down(e, recursive) {
    var _this2 = this;

    if (e.persist) {
      e.persist();
    }
    this.stop();
    this.step('down', e);
    this.autoStepTimer = setTimeout(function () {
      _this2.down(e, true);
    }, recursive ? SPEED : DELAY);
  },
  up: function up(e, recursive) {
    var _this3 = this;

    if (e.persist) {
      e.persist();
    }
    this.stop();
    this.step('up', e);
    this.autoStepTimer = setTimeout(function () {
      _this3.up(e, true);
    }, recursive ? SPEED : DELAY);
  }
};
module.exports = exports['default'];