'use strict';

exports.__esModule = true;
exports.default = scssTokenize;
var SINGLE_QUOTE = 39;
var DOUBLE_QUOTE = 34;
var BACKSLASH = 92;
var SLASH = 47;
var NEWLINE = 10;
var SPACE = 32;
var FEED = 12;
var TAB = 9;
var CR = 13;
var OPEN_PARENTHESES = 40;
var CLOSE_PARENTHESES = 41;
var OPEN_CURLY = 123;
var CLOSE_CURLY = 125;
var SEMICOLON = 59;
var ASTERICK = 42;
var COLON = 58;
var AT = 64;
var COMMA = 44;
var HASH = 35;

var RE_AT_END = /[ \n\t\r\f\{\(\)'"\\;/#]/g;
var RE_NEW_LINE = /[\r\f\n]/g;
var RE_WORD_END = /[ \n\t\r\f\(\)\{\}:;@!'"\\#]|\/(?=\*)/g;
var RE_BAD_BRACKET = /.[\\\/\("'\n]/;

function scssTokenize(input) {
    var tokens = [];
    var css = input.css.valueOf();

    var code = void 0,
        next = void 0,
        quote = void 0,
        lines = void 0,
        last = void 0,
        content = void 0,
        escape = void 0,
        nextLine = void 0,
        nextOffset = void 0,
        escaped = void 0,
        escapePos = void 0,
        prev = void 0,
        n = void 0,
        brackets = void 0;

    var length = css.length;
    var offset = -1;
    var line = 1;
    var pos = 0;

    function unclosed(what) {
        throw input.error('Unclosed ' + what, line, pos - offset);
    }

    while (pos < length) {
        code = css.charCodeAt(pos);

        if (code === NEWLINE || code === FEED || code === CR && css.charCodeAt(pos + 1) !== NEWLINE) {
            offset = pos;
            line += 1;
        }

        switch (code) {
            case NEWLINE:
            case SPACE:
            case TAB:
            case CR:
            case FEED:
                next = pos;
                do {
                    next += 1;
                    code = css.charCodeAt(next);
                    if (code === NEWLINE) {
                        offset = next;
                        line += 1;
                    }
                } while (code === SPACE || code === NEWLINE || code === TAB || code === CR || code === FEED);

                tokens.push(['space', css.slice(pos, next)]);
                pos = next - 1;
                break;

            case OPEN_CURLY:
                tokens.push(['{', '{', line, pos - offset]);
                break;

            case CLOSE_CURLY:
                tokens.push(['}', '}', line, pos - offset]);
                break;

            case COLON:
                tokens.push([':', ':', line, pos - offset]);
                break;

            case SEMICOLON:
                tokens.push([';', ';', line, pos - offset]);
                break;

            case COMMA:
                tokens.push(['word', ',', line, pos - offset, line, pos - offset + 1]);
                break;

            case OPEN_PARENTHESES:
                prev = tokens.length ? tokens[tokens.length - 1][1] : '';
                n = css.charCodeAt(pos + 1);
                if (prev === 'url' && n !== SINGLE_QUOTE && n !== DOUBLE_QUOTE && n !== SPACE && n !== NEWLINE && n !== TAB && n !== FEED && n !== CR) {

                    brackets = 1;
                    escaped = false;
                    next = pos + 1;
                    while (next <= css.length - 1) {
                        n = css.charCodeAt(next);
                        if (n === BACKSLASH) {
                            escaped = !escaped;
                        } else if (n === OPEN_PARENTHESES) {
                            brackets += 1;
                        } else if (n === CLOSE_PARENTHESES) {
                            brackets -= 1;
                            if (brackets === 0) break;
                        }
                        next += 1;
                    }

                    tokens.push(['brackets', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                    pos = next;
                } else {
                    next = css.indexOf(')', pos + 1);
                    content = css.slice(pos, next + 1);

                    if (next === -1 || RE_BAD_BRACKET.test(content)) {
                        tokens.push(['(', '(', line, pos - offset]);
                    } else {
                        tokens.push(['brackets', content, line, pos - offset, line, next - offset]);
                        pos = next;
                    }
                }

                break;

            case CLOSE_PARENTHESES:
                tokens.push([')', ')', line, pos - offset]);
                break;

            case SINGLE_QUOTE:
            case DOUBLE_QUOTE:
                quote = code === SINGLE_QUOTE ? '\'' : '"';
                next = pos;
                do {
                    escaped = false;
                    next = css.indexOf(quote, next + 1);
                    if (next === -1) unclosed('quote');
                    escapePos = next;
                    while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
                        escapePos -= 1;
                        escaped = !escaped;
                    }
                } while (escaped);

                content = css.slice(pos, next + 1);
                lines = content.split('\n');
                last = lines.length - 1;

                if (last > 0) {
                    nextLine = line + last;
                    nextOffset = next - lines[last].length;
                } else {
                    nextLine = line;
                    nextOffset = offset;
                }

                tokens.push(['string', css.slice(pos, next + 1), line, pos - offset, nextLine, next - nextOffset]);

                offset = nextOffset;
                line = nextLine;
                pos = next;
                break;

            case AT:
                RE_AT_END.lastIndex = pos + 1;
                RE_AT_END.test(css);
                if (RE_AT_END.lastIndex === 0) {
                    next = css.length - 1;
                } else {
                    next = RE_AT_END.lastIndex - 2;
                }
                tokens.push(['at-word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                pos = next;
                break;

            case BACKSLASH:
                next = pos;
                escape = true;
                while (css.charCodeAt(next + 1) === BACKSLASH) {
                    next += 1;
                    escape = !escape;
                }
                code = css.charCodeAt(next + 1);
                if (escape && code !== SLASH && code !== SPACE && code !== NEWLINE && code !== TAB && code !== CR && code !== FEED) {
                    next += 1;
                }
                tokens.push(['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                pos = next;
                break;

            default:
                n = css.charCodeAt(pos + 1);

                if (code === HASH && n === OPEN_CURLY) {
                    var deep = 1;
                    next = pos;
                    while (deep > 0) {
                        next += 1;
                        if (css.length <= next) unclosed('interpolation');

                        code = css.charCodeAt(next);
                        n = css.charCodeAt(next + 1);

                        if (code === CLOSE_CURLY) {
                            deep -= 1;
                        } else if (code === HASH && n === OPEN_CURLY) {
                            deep += 1;
                        }
                    }

                    content = css.slice(pos, next + 1);
                    lines = content.split('\n');
                    last = lines.length - 1;

                    if (last > 0) {
                        nextLine = line + last;
                        nextOffset = next - lines[last].length;
                    } else {
                        nextLine = line;
                        nextOffset = offset;
                    }

                    tokens.push(['word', content, line, pos - offset, nextLine, next - nextOffset]);

                    offset = nextOffset;
                    line = nextLine;
                    pos = next;
                } else if (code === SLASH && n === ASTERICK) {
                    next = css.indexOf('*/', pos + 2) + 1;
                    if (next === 0) unclosed('comment');

                    content = css.slice(pos, next + 1);
                    lines = content.split('\n');
                    last = lines.length - 1;

                    if (last > 0) {
                        nextLine = line + last;
                        nextOffset = next - lines[last].length;
                    } else {
                        nextLine = line;
                        nextOffset = offset;
                    }

                    tokens.push(['comment', content, line, pos - offset, nextLine, next - nextOffset]);

                    offset = nextOffset;
                    line = nextLine;
                    pos = next;
                } else if (code === SLASH && n === SLASH) {
                    RE_NEW_LINE.lastIndex = pos + 1;
                    RE_NEW_LINE.test(css);
                    if (RE_NEW_LINE.lastIndex === 0) {
                        next = css.length - 1;
                    } else {
                        next = RE_NEW_LINE.lastIndex - 2;
                    }

                    content = css.slice(pos, next + 1);

                    tokens.push(['comment', content, line, pos - offset, line, next - offset, 'inline']);

                    pos = next;
                } else {
                    RE_WORD_END.lastIndex = pos + 1;
                    RE_WORD_END.test(css);
                    if (RE_WORD_END.lastIndex === 0) {
                        next = css.length - 1;
                    } else {
                        next = RE_WORD_END.lastIndex - 2;
                    }

                    tokens.push(['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                    pos = next;
                }

                break;
        }

        pos++;
    }

    return tokens;
}
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;base64,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
