const fs = require('fs')
const os = require('os')
const path = require('path')
const mkdirp = require('mkdirp')
const sliced = require('sliced')
const once = require('once')
const debug = require('debug')('avatar:main')
const debugSource = require('debug')('avatar:source')
const Events = require('events')

const { execute, inject } = require('./javascript')
const actions = require('./actions')
const preload = require('./preload')

const noop = function () {}
const keys = Object.keys
const DEFAULT_HALT_MESSAGE = 'Avatar Halted'


const datadir = path.join(os.tmpdir(), 'avatar')
const preloadFile = path.join(datadir, 'preload.js')

mkdirp.sync(datadir)
fs.writeFileSync(preloadFile, `(${preload.toString()})()`)

const container = document.createElement('div')
container.style.width = container.style.height = 0
container.style.overflow = 'hidden'
document.body.appendChild(container)

// wrap all the functions in the queueing function
function queued (name, fn) {
    return function action () {
        debug('queueing action "' + name + '"')
        var args = [].slice.call(arguments)
        this._queue.push([fn, args])
        return this
    }
}

window.__avatars__ = {};

let n = 0;

class Avatar extends Events {
    constructor (options = {}) {
        super();

        this.state = 'initial';
        this.running = false;
        this.ending = false;
        this.ended = false;
        this._queue = [];
        this._headers = [];

        const defaultOptions = {
            waitTimeout: 5 * 60 * 1000, // wait超时时间
            gotoTimeout: 30 * 1000, // 页面加载超时时间
            pollInterval: 250, // wait检测间隔
            typeInterval: 100, // type间隔
            executionTimeout: 30 * 1000, // 执行超时时间
            Promise: Avatar.Promise || Promise,
            container,
            url: 'about:blank'
        }

        window.localStorage.debug = '';
        this.nid = n++;
        options = this.options = Object.assign({}, defaultOptions, options);
        this.id = options.id || this.nid;
        if (options.debug) {
            window.localStorage.debug = options.debug;
        }
        const webview = this.options.webview || document.createElement('webview');
        webview.style.height = webview.style.width = '100%';
        this.webview = webview;

        const self = this;
        function forward (name) {
            return function (event) {
                if (!self._closed) {
                    self.emit(name, event);
                }
            };
        }

        webview.addEventListener('did-finish-load', forward('did-finish-load'));
        webview.addEventListener('did-fail-load', forward('did-fail-load'));
        webview.addEventListener('did-fail-provisional-load', forward('did-fail-provisional-load'));
        webview.addEventListener('did-frame-finish-load', forward('did-frame-finish-load'));
        webview.addEventListener('did-start-loading', forward('did-start-loading'));
        webview.addEventListener('did-stop-loading', forward('did-stop-loading'));
        webview.addEventListener('did-get-response-details', forward('did-get-response-details'));
        webview.addEventListener('did-get-redirect-request', forward('did-get-redirect-request'));
        webview.addEventListener('dom-ready', forward('dom-ready'));
        webview.addEventListener('page-favicon-updated', forward('page-favicon-updated'));
        webview.addEventListener('new-window', forward('new-window'));
        webview.addEventListener('will-navigate', forward('will-navigate'));
        webview.addEventListener('crashed', forward('crashed'));
        webview.addEventListener('plugin-crashed', forward('plugin-crashed'));
        webview.addEventListener('destroyed', forward('destroyed'));
        webview.addEventListener('media-started-playing', forward('media-started-playing'));
        webview.addEventListener('media-paused', forward('media-paused'));
        webview.addEventListener('close', () => {
            this._closed = true;
            forward('close');
        });

        debug('queuing process start');
        this._ready = new options.Promise(resolve => {
            function didDomReady () {
                webview.removeEventListener('dom-ready', didDomReady);
                resolve();
            }
            webview.addEventListener('dom-ready', didDomReady);
            webview.setAttribute('partition', `persist:${this.id}`);
            webview.setAttribute('src', options.url);
            webview.setAttribute('disablewebsecurity', true);
            webview.setAttribute('preload', `file://${preloadFile}`);
            options.container.appendChild(webview);
        });
        this.queue(done => {
            debug(`${this.id} start`);
            this._ready.then(() => {
                debug(`${this.id} ready`);
                done();
            });
        });
        Avatar.namespaces.forEach(function (name) {
            if (typeof this[name] === 'function') {
                this[name] = this[name]();
            }
        }, this);
        // TODO:
        window.__avatars__[`${this.id}_${this.nid}`] = this;
    }
    _end (fn) {
        delete window.__avatars__[`${this.id}_${this.nid}`];
        if (this.webview) {
            this.webview.stop();
            this.webview.remove();
            this.webview = null;
        }
        this.ended = true;
        fn()
    }
    destroy () {
        this.halt();
    }
    run (fn) {
        debug('running');
        var steps = this.queue();
        this.running = true;
        this._queue = [];
        this.queue((done) => {
            this._ready.then(() => {
                done();
            });
        });
        var self = this;

        // kick us off
        next();

        // next function
        function next (err, res) {
            var item = steps.shift();
            // Immediately halt execution if an error has been thrown, or we have no more queued up steps.
            if (err || !item) return done.apply(self, arguments);
            var args = item[1] || [];
            var method = item[0];
            args.push(once(after));
            method.apply(self, args);
        }

        function after (err, res) {
            err = err || self.die;
            var args = sliced(arguments);
            next.apply(self, args);
        }

        function done () {
            debug('run success');
            var doneargs = arguments;
            self.running = false;
            if (self.ending) {
                self._end(() => fn.apply(self, doneargs))
            }
            return fn.apply(self, doneargs);
        }

        return this;
    }
    queue (done) {
        if (!arguments.length) return this._queue;
        var args = sliced(arguments);
        var fn = args.pop();
        this._queue.push([fn, args]);
    }
    end (done) {
        this.ending = true;

        if (done && !this.running && !this.ended) {
            return this.then(done);
        }

        return this;
    }
    halt (error, done) {
        this.ending = true;
        var queue = this.queue(); // empty the queue
        queue.splice(0);
        if (!this.ended) {
            var message = error;
            if (error instanceof Error) {
                message = error.message;
            }
            this.die = message || DEFAULT_HALT_MESSAGE;
            if (typeof this._rejectActivePromise === 'function') {
                this._rejectActivePromise(error || DEFAULT_HALT_MESSAGE);
            }
            var callback = done;
            if (!callback || typeof callback !== 'function') {
                callback = noop;
            }
            this._end(callback);
        }
        return this;
    }
    then (fulfill, reject) {
        return new this.options.Promise((success, failure) => {
            this._rejectActivePromise = failure;
            this.run(function (err, result) {
                if (err) failure(err);
                else success(result);
            });
        })
        .then(fulfill, reject);
    }
    catch (reject) {
        this._rejectActivePromise = reject;
        return this.then(undefined, reject);
    }
    header (header, value) {
        if (header && typeof value !== 'undefined') {
            this._headers[header] = value;
        } else {
            this._headers = header || {};
        }
        return this;
    }
    _inject (js, done) {
        const {webview} = this;
        const source = inject({src: js});
        debugSource(`inject source: ${source}`);
        webview.executeJavaScript(source, response => {
            if (response.err) {
                return done(response.err);
            }
            done(null, response.result);
        });
        return this;
    }
    evaluateNow (fn, done) {
        const {webview} = this;
        if (!webview.executeJavaScript) throw new Error('webview do not ready, can not call executeJavaScript.');
        const args = Array.prototype.slice.call(arguments).slice(2).map(a => {
            return {argument: JSON.stringify(a)};
        });
        const source = execute({src: String(fn), args: args});
        debugSource(`evaluate source: ${source}`);
        webview.executeJavaScript(source, response => {
            if (response.err) {
                return done(response.err);
            }
            done(null, response.result);
        });
        return this;
    }
    use (fn) {
        fn(this)
        return this
    }
}

Avatar.namespaces = [];
Avatar.Promise = Promise;
Avatar.action = (name, fn) => {
    if (typeof fn === 'function') {
        Avatar.prototype[name] = queued(name, fn);
    } else {
        if (!~Avatar.namespaces.indexOf(name)) {
            Avatar.namespaces.push(name);
        }
        Avatar.prototype[name] = function () {
            var self = this;
            return keys(fn).reduce(function (obj, key) {
                obj[key] = queued(name, fn[key]).bind(self);
                return obj;
            }, {});
        };
    }
}

Object.keys(actions).forEach(name => {
    var fn = actions[name];
    Avatar.action(name, fn);
});

module.exports = Avatar;
